/**
 *
 * \file        usb_console.c
 *
 * \brief       Implementation of a console that uses the USB device port.
 *
 * \author      Pete McCormick
 *
 * \date        5/6/2008
 *
 * \note        This is a generic file.  There should be NO product-specific
 *              nor processor-specific stuff in here.
 *
 */

////////////////////////////////////////////////////////////////////////////////

#include "console.h"
#include "os.h"
#include "string_utils.h"
#include <string.h>
#include "hardware.h"
#include "usb_device.h"
#include "device.h"
#include <stdio.h>


////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////

extern "C" void USBConsoleInit(void);

USBCONSOLE USBConsole;

////////////////////////////////////////////////////////////////////////////////

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Unused for now - needed if we ever implement
 *              XGET, XPUT, or SYSTEM commands.
 *
 * \param       void
 *
 */
void USBConsoleCaptureInput(void)
{
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Unused for now - needed if we ever implement
 *              XGET, XPUT, or SYSTEM commands.
 *
 * \param       void
 *
 */
void USBConsoleReleaseInput(void)
{
}

/**
 * \author      Jeff Paige
 *
 * \date        12/26/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Prints the welcome message
 *
 * \param       not used
 *
 */
void USBConsoleDisplayWelcomeMessage(void)
{
    CMDINPUT * pCmdInput = &USBConsole.cmdInput;
    DmConsoleDisplayWelcome(pCmdInput);
}

/**
 * \author      Jeff Paige
 *
 * \date        12/26/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       RX data from teh USB port
 *
 * \param       pData - pointer to data buffer
 *              dataBytes - number of bytes
 *
 */
INT32 USBConsoleRxData(UINT32 param, UINT8 * pData, UINT32 dataBytes)
{
    CMDINPUT * pCmdInput = &USBConsole.cmdInput;

    // Xmodem capture mode.
    if ( pCmdInput->captured )
    {
        //check if this USB console supports xmodem captures
        if ( pfPTNQXmodemPacket )
        {
            (*pfPTNQXmodemPacket)((unsigned char*)pData, dataBytes);
            return 0;
        }

    }

   if (pCmdInput->pPassto)
   {
       pCmdInput->pPassto->pSendFunc(pData, dataBytes, pCmdInput->pPassto->sendParam);
       return 0;
   }

    // make sure last command has been processed since buffer is shared
    if ( DmConsoleWaitBufferAvailable(pCmdInput, MAX_CONSOLE_WAIT) )
    {
        while (dataBytes--)
        {
            DmConsoleBufferInput(pCmdInput, *pData++);
        }
    }

    return 0;
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Sets up structures used for the UART console.
 *
 * \param       void
 *
 * \note        Runs once at startup time.
 *
 */
void USBConsoleInit(void)
{
    CMDINPUT * pCmdInput = &USBConsole.cmdInput;

    // needed when actually parsing the command
    pCmdInput->command.pCmdInput = pCmdInput;
    pCmdInput->pSendFunc = USBConsoleTx;
    pCmdInput->sendParam = 0;
    pCmdInput->pCaptureFunc = USBConsoleCaptureInput;
    pCmdInput->pReleaseFunc = USBConsoleReleaseInput;
    pCmdInput->cmdSource = FROM_USB_CONSOLE;
    pCmdInput->printLock = 0;
    pCmdInput->captured  = 0;
    pCmdInput->TjiPrintLock = 0;
    pCmdInput->consoleLock = 0;
    pCmdInput->pPassto = NULL;

    CmdInputAddToList(pCmdInput);
}

/**
* \author      S.Novick
* \date        04/22/2013
* \brief       Kills USB console
* \param       none
* \return      void
* \retval      none
*/
void USBConsoleCloseInput( void )
{
    CMDINPUT * pCmdInput = &USBConsole.cmdInput;

    if (pCmdInput->pPassto)
    {
        pCmdInput->pPassto->pPassto = NULL;
    }

    DmConsoleCloseInput( &USBConsole.cmdInput );
}


